<?php

/**
 * OddsscannerSkinLoader
 *
 * This class serves as a companion for the "OddsScanner" skin. 
 * It automatically loads all JavaScript files located in the "scripts" 
 * folder inside the skin directory and injects them into the HTML 
 * before the closing </body> tag.
 * 
 * To enable this plugin, edit server.php file and add "OddsscannerSkinLoader" to AVAILABLE_PLUGINS.
 * e.g.: 
 * define('AVAILABLE_PLUGINS', 'ext,OddsscannerSkinLoader');
 */

class OddsscannerSkinLoader{
    private $skinPath;
    private $scriptsFolder;

    public function __construct(){
        $this->skinPath = '/css/skins/OddsScanner';
        $this->scriptsFolder = '/scripts';

        // Hook into output buffer to inject scripts before </head>
        ob_start([$this, 'injectScripts']);
    }

    private function checkAndLoadScripts(){
        $absoluteSkinPath = dirname(__DIR__, 3) . $this->skinPath;

        if (is_dir($absoluteSkinPath)) {
            return $this->getJavaScriptTags($absoluteSkinPath);
        }

        return '';
    }

    private function getJavaScriptTags($path){
        $jsFiles = glob($path . $this->scriptsFolder . '/*.js');
        $scriptTags = '';

        foreach ($jsFiles as $file) {
            // Get the base URL dynamically
            $baseURL = $this->getBaseURL();

            $fileUrl = str_replace(dirname(__DIR__, 3), '', $file);
            $fileUrl = $baseURL . $fileUrl; 
                        
            $scriptTags .= "<script defer src='$fileUrl'></script>\n";
        }

        return $scriptTags;
    }

    private function getBaseURL(){
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? "https" : "http";
        $host = $_SERVER['HTTP_HOST']; 
        $scriptPath = $_SERVER['SCRIPT_NAME']; 

        $basePath = (strpos($scriptPath, '/development') !== false) ? '/development' : '';

        if ($host){
            return "$protocol://$host$basePath";
        } else {
            return "$protocol://$basePath";
        }
        
    }

    public function injectScripts($buffer){
        $scripts = $this->checkAndLoadScripts();

        return str_replace('</body>', $scripts . '</body>', $buffer);
    }
}

new OddsscannerSkinLoader();