<?php

require 'vendor/autoload.php';

use Google\Cloud\BigQuery\BigQueryClient;
header('Content-Type: application/json');
// Read the JSON config file
$keyFilePath = 'config.json';
$configJson = file_get_contents($keyFilePath);
$configData = json_decode($configJson, true);

// Extract the project ID
$projectId = $configData['project_id'] ?? null;

if (!$projectId) {
    fwrite(STDERR, "Error: Project ID not found in config.json\n");
    exit(1);
}


// --- Configuration ---
$allowedTables = [
    'phoenix_sandbox' => [
        'affiliate_program_accounts_info',
        'affiliate_program_accounts_monthly_info',
        'bi_config_connections'
    ]
];

// --- Command line arguments ---
if ($argc !== 4) {
    fwrite(STDERR, "Usage: php bigquery_query.php <dataset> <table> <limit>\n");
    exit(1);
}

if (!isset($_GET['dataset']) || !isset($_GET['table']) || !isset($_GET['limit'])) {
    die(json_encode(["error" => "Missing required parameters."]));
}

$dataset = $_GET['dataset'] ?? '';
$table = $_GET['table'] ?? '';
$limit = $_GET['limit'] ?? '';

// --- Validation ---
if (!array_key_exists($dataset, $allowedTables) || !in_array($table, $allowedTables[$dataset])) {
    fwrite(STDERR, "Invalid dataset or table.\n");
    exit(1);
}


try {
    $bigQuery = new BigQueryClient([
        'keyFilePath' => $keyFilePath
    ]);

    $fullTableId = "{$projectId}.{$dataset}.{$table}";
    $query = "SELECT * FROM `{$fullTableId}` LIMIT {$limit}";

    $queryJobConfig = $bigQuery->query($query);
    $queryResults = $bigQuery->runQuery($queryJobConfig);

    if (!$queryResults->isComplete()) {
        throw new Exception("Query did not complete.");
    }

    $rows = [];
    foreach ($queryResults->rows() as $row) {
        $rows[] = $row;
    }

    echo json_encode($rows);

} catch (Exception $e) {
    fwrite(STDERR, "Error: " . $e->getMessage() . "\n");
    exit(1);
}
