// @Moment.js Default Timezone - UTC
moment.tz.setDefault("UTC");


// @Constraints
const TABLE_INFO = {
    day: {
        title: 'Day',
        info: 'Reference Date.',
    },
    month: {
        title: 'Month',
        info: 'Reference Date.',
    },
    clicks: {
        title: 'Clicks',
        info: 'The total number of times users have clicked on your affiliate tracking links.'
    },
    signUps: {
        title: 'Signups',
        info: 'The number of users who register an account with the operator after clicking your link.'
    },
    ftds: {
        title: 'FTDs',
        info: 'The number of players who make their first deposit after signing up.'
    },
    cpas: {
        title: 'CPAs',
        info: 'The number of players that triggered the commission after depositing baseline or more.'
    },
    deposits: {
        title: 'Deposits',
        info: 'The total amount of money that your referred players have deposited into their accounts with the operator.'
    },
    ngr: {
        title: 'NGR',
        info: 'Total net revenue of your referred players.'
    },
    totalComissions: {
        title: 'Total Commissions',
        info: 'The amount generated as a sum of CPA and RS earnings.'
    },
    advertiser: {
        title: 'Advertiser',
        // info: '',
    },
    trafficSource: {
        title: 'Traffic Source',
        // info: '',
    },
    period: {
        title: 'Period',
        // info: '',
    },
    cpaCommissions: {
        title: 'CPA Commissions',
        info: 'The total payout you earn for each new player who meets baseline (and in some cases wagering) criteria.',
    },
    rsCommissions: {
        title: 'RS Commissions',
        info: 'A percentage of the operator’s Net Gaming Revenue (NGR) from the players you refer (T&Cs apply).',
    },
    subAccount: {
        title: 'Sub-affiliate',
        // info: '',
    },
    kycStatus: {
        title: 'KYC Status',
        info: 'The current documentation approval status of your sub-affiliate.',
    },
    paymentFrequency: {
        title: 'Payment Frequency',
        // info: '',
    },
    periodEarnings: {
        title: 'Period',
        info: 'The selected time frame of earnings related to sub-affiliates.',
    }
}

const METRICS_INFO = {
    clicks: {
        title: 'Clicks',
        info: 'The total number of times users have clicked on your affiliate tracking links.',
    },
    signUps: {
        title: 'Signups',
        info: 'The number of users who register an account with the operator after clicking your link.',
    },
    ftds: {
        title: 'FTDs',
        info: 'The number of players who make their first deposit after signing up.',
    },
    cpas: {
        title: 'CPAs',
        info: 'The number of players that triggered the commission after depositing baseline or more.',
    },
    totalNGR: {
        title: 'Total Amount of NGR',
        info: 'Total Amount of NGR for the selected dates.',
    },
    totalDeposits: {
        title: 'Total Amount of Deposits',
        info: 'Total Amount of Deposits for the selected dates.',
    },
    totalCommissions: {
        title: 'Total Commissions',
        info: 'The amount generated as a sum of CPA and RS earnings.',
    },
    cpaCommissions: {
        title: 'CPA Commissions',
        info: 'The total payout you earn for each new player who meets baseline (and in some cases wagering) criteria.',
    },
    rsCommissions: {
        title: 'RS Commissions',
        info: 'A percentage of the operator’s Net Gaming Revenue (NGR) from the players you refer (T&Cs apply).',
    },
}

const ERROR_HTML = `<div class='report-page-error'><p>Error to fetch the data</p></div>`;

const BASE_PATH = window.location.origin + getBasePath();
const MIDDLEWARE_URL = BASE_PATH + '/index.php?module=OddsscannerSkinLoader/middleware/index&endpoint=';

const enUS = {
    days: ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thurday', 'Friday', 'Saturday'],
    daysShort: ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'],
    daysMin: ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'],
    months: ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'],
    monthsShort: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
    today: 'Today',
    clear: 'Clean',
    dateFormat: 'dd/MM/yyyy',
    timeFormat: 'HH:mm',
    firstDay: 0
};

const STATUS_MAP = {
    'Accepted': 'accepted',
    'In Progress': 'in-progress',
    'On Hold': 'on-hold',
    'Pending': 'pending',
    'Pending KYC Documents': 'pending-documents',
    'Pending Payment Information': 'pending-information',
    'Rejected': 'rejected',
    'Submitted': 'submitted',
}


// @Aux - Constraints
function getBasePath() {
    const path = window.location.pathname;
    if (path.includes('/prelaunch/')) return '/prelaunch';
    if (path.includes('/development/')) return '/development';
    return '';
}

function getRukoUserID() {
    const userID = $('.ruko-user-id').val();

    if (!BASE_PATH.includes('partners.rukovoditel.xyz')) return userID;

    let customID = userID;

    if (Number(userID) === 864) customID = 29;
    else if (Number(userID) === 874) customID = 673;
    else if (Number(userID) === 875) customID = 446
    else if (Number(userID) === 876) customID = 202;
    else if (Number(userID) === 891) customID = 963;
    else if (Number(userID) === 892) customID = 325;
    else if (Number(userID) === 1000) customID = 779; // Master IDs: 963, 779, 325

    return customID;
}



// @Chart
function formatRecordsChart(records) {
    const formattedRecords = {};

    formattedRecords.labels = records.map((record) => {
        const [day, month, year] = record.day.split('/');
        const date = new Date(Date.UTC(year, month - 1, day));

        return date.toLocaleDateString('en-US', {
            month: 'short',
            day: '2-digit',
            timeZone: 'UTC',
        });
    });

    formattedRecords.datasets = [
        {
            label: 'Clicks',
            data: records.map((record) => record.clicks),
            borderColor: 'rgba(156, 54, 181, 0.5)',
            backgroundColor: '#9C36B5',
            pointBorderColor: 'rgba(156, 54, 181, 0.25)',
            borderWidth: 4,
            radius: 4,
        },
        {
            label: 'FTDs',
            borderColor: 'rgba(59, 91, 219, 0.5)',
            backgroundColor: '#3B5BDB',
            pointBorderColor: 'rgba(59, 91, 219, 0.25)',
            data: records.map((record) => record.ftds),
            borderWidth: 4,
            radius: 4,
        },
        {
            label: 'Commisions',
            borderColor: 'rgba(247, 103, 7, 0.5)',
            backgroundColor: '#F76707',
            pointBorderColor: 'rgba(247, 103, 7, 0.25)',
            data: records.map((record) => record.totalCommissions),
            borderWidth: 4,
            radius: 4,
        }
    ]

    return formattedRecords;
}

function loadChart(chartType, data) {
    const formattedRecords = formatRecordsChart(data);
    const ctx = document.getElementById('report_canvas').getContext('2d');

    return new Chart(ctx, {
        type: chartType,
        data: formattedRecords,
        options: {
            layout: { padding: 10 },
            plugins: {
                legend: {
                    position: 'top',
                    align: 'end',
                    labels: {
                        font: {
                            size: 14,
                            weight: 'bold',
                        },
                        color: '#374151',
                        boxWidth: 20,
                        padding: 16,
                        useBorderRadius: true,
                    },
                },
                title: {
                    display: false,
                    text: 'Clicks by date and advertiser',
                },
            },
            scales: {
                y: {
                    title: { display: false, text: 'y label' },
                },
                x: {
                    title: { display: false, text: 'x label', }
                },
            },
            responsive: true,
            onResize: (chartInstance, newSize) => {
                const canvas = chartInstance.canvas;
                const container = canvas.parentNode;
                const minWidth = 624;
                const originalAspect = 568 / 1136;

                if (newSize.width <= minWidth) {
                    const targetWidth = minWidth;
                    const targetHeight = Math.round(minWidth * originalAspect);

                    container.style.width = `${targetWidth}px`;
                    container.style.height = `${targetHeight}px`;

                    canvas.style.width = `${targetWidth}px`;
                    canvas.style.height = `${targetHeight}px`;

                    chartInstance.options.plugins.legend.align = 'start';
                } else {
                    container.style.width = '';
                    container.style.height = '';
                    canvas.style.width = '';
                    canvas.style.height = '';

                    chartInstance.options.plugins.legend.align = 'end';
                }

                chartInstance.update();
            },
        }
    });
}



// @Reports
function createReport(id, rukoUserId) {
    let content = '';

    if (id === 'tab-traffic-report') content = 'Analyze traffic volume, quality, and behavior to optimize your acquisition strategies.';
    if (id === 'tab-advertiser-report') content = 'Get detailed insights about your performance per advertiser and conversion efficiency.';
    if (id === 'tab-traffic-source-report') content = 'Compare your traffic sources to identify the best performers and maximize your ROI.';
    if (id === 'tab-earnings-report') content = 'Keep on track your CPA and RS and have a clear view of your total earnings.';
    if (id === 'tab-network-traffic') content = 'Understand performance metrics by sub-affiliate and advertiser, enabling detailed tracking of traffic and conversions in digital campaigns.';
    if (id === 'tab-network-earnings') content = 'Get overview of the earnings generated by your sub-affiliates, breaking down how in different periods and through different commission types.';

    $report = `
        <div class='report-item' id='${id}'>
            <div class='top'>
                <div class='content'>
                    <p>${content}</p>
                </div>

                <div class='filters-container'>
                    <div class='filters-mobile'>
                        <div class='icon-filter'></div>
                        
                        <div class='title'>
                            <span class='content'>Filter results </span>
                            <span class='counter'>(1 filter applied)</span>
                        </div>

                        <div class='icon-arrow'></div>
                    </div>

                    <div class='report-item-filters'>
                        <div class='filters-menu' id=''></div>
                    </div>
                </div>
            </div>

            <div class='bottom'></div>
        </div>
    `

    $('#report_page_content .report-page-main').append($report);

    const reportItem = $('#report_page_content').find(`.report-page-main .report-item#${id}`);
    if (!reportItem.length) return null;

    showReport(reportItem);

    $('#report_page_content').find('.report-item.active .filters-mobile')
        .on('click', handleFiltersMobileClick)

    loadReportData(id, rukoUserId);
}

function showReport(reportItem) {
    $('#report_page_content').find('.report-page-main .report-item.active').not(reportItem).removeClass('active');
    $(reportItem).toggleClass('active');
}



// @Tabs Navigation
function changeTab(e, rukoUserId) {
    const tab = $(e.currentTarget);

    if ($(tab).hasClass('active')) return;

    $('#report_page_content').find('.report-page-tabs .tab.active').not(tab).removeClass('active');
    $(tab).toggleClass('active');

    const id = $(tab).data('id');

    const reportItem = $('#report_page_content').find(`.report-page-main .report-item#${id}`);

    if (!reportItem.length) {
        createReport(id, rukoUserId);
        return;
    }

    showReport(reportItem);
}



// @Static Loading
function loadPage() {
    const col = $('.page-content').children('.row').children('.col-md-12');
    const container = $('<div class="report-container"></div>');

    $(col).append(container);
    $(col).children('.row').appendTo(container);

    const firstRow = $('.report-container > .row').first();
    const lastRow = $('.report-container > .row').last();

    $(firstRow).addClass('top');
    $(lastRow).addClass('bottom');

    $(firstRow).children().removeClass('col-md-11 col-md-1 align-right');
    $(firstRow).children().last().hide();

    $(firstRow).find('.page-title').addClass('page-title-reports');

    $('#report_page_content').append("<div class='report-page-container'><div class='loader'></div></div>");
}

function loadMain() {
    $main = `
        <div class='report-page-main'>
            <div class='report-item active' id='tab-summary'>
                <div class='top'>
                    <div class='content'>
                        <p>Here you get a quick overview of all your metrics.</p>
                    </div>

                    <div class='filters-container'>
                        <div class='filters-mobile'>
                            <div class='icon-filter'></div>
                            
                            <div class='title'>
                                <span class='content'>Filter results </span>
                                <span class='counter'>(1 filter applied)</span>
                            </div>

                            <div class='icon-arrow'></div>
                        </div>

                        <div class='report-item-filters'>
                            <div class='filters-menu' id=''></div>
                        </div>
                    </div>
                </div>

                <div class='bottom'></div>
            </div>            
        </div>
    `
    $('#report_page_content .report-page-container').append($main);

    $('#report_page_content').find('.report-item.active .filters-mobile')
        .on('click', handleFiltersMobileClick);
}

function loadTabs(rukoUserId, IsMasterAccount = false) {
    $tabs = `
        <div class='report-page-tabs'>
            <div class='tab active' data-id='tab-summary'>
                <span>Summary</span>
            </div>

            <div class='tab' data-id='tab-traffic-report'>
                <span>Traffic Report</span>
            </div>

            <div class='tab' data-id='tab-advertiser-report'>
                <span>Advertiser Report</span>
            </div>

            <div class='tab' data-id='tab-traffic-source-report'>
                <span>Traffic Source Report</span>
            </div>

            <div class='tab' data-id='tab-earnings-report'>
                <span>Earnings Report</span>
            </div>
    `

    if (IsMasterAccount) $tabs += `
        <div class='tab' data-id='tab-network-traffic'>
            <span>Network Traffic</span>
        </div>

        <div class='tab' data-id='tab-network-earnings'>
            <span>Network Earnings</span>
        </div>
    `

    $tabs += '</div>'

    $('#report_page_content .report-page-container').append($tabs);

    $('#report_page_content').find('.report-page-tabs .tab')
        .on('click', (e) => changeTab(e, rukoUserId))
}

function loadPeriodSelectorSummary() {
    const firstDay = moment().startOf('month').format('YYYY-MM-DD');
    const today = moment().format('YYYY-MM-DD');

    $filter = `
        <div class='filter summary-period-selector active'>
            <div class='active-option' data-name='period-selector' data-time='Daily' data-type='month-to-date' data-value='{"startDate":"${firstDay}","endDate":"${today}"}'>
                <div class='title'>
                    <span>Month to date</span>
                </div>

                <div class='icons'>
                    <div class='icon-toggle'></div>
                </div>
            </div>

            <div class='wrapper'>
                <div class='options select-single'>
                    <div class='option active' data-value='month-to-date'>
                        <div class='icon-select'></div>
                        <span>Month to date</span>
                    </div>

                    <div class='option custom-date' data-value='custom-date'>
                        <div class='icon-select'></div>
                        <span>Custom Date</span>
                    </div>

                    <div id="calendar-full-summary-mobile" class="calendar mobile"></div>

                    <div class='option' data-value='yesterday'>
                        <div class='icon-select'></div>
                        <span>Yesterday</span>
                    </div>

                    <div class='option' data-value='last-month'>
                        <div class='icon-select'></div>
                        <span>Last Month</span>
                    </div>

                    <div class='option' data-value='last-7-days'>
                        <div class='icon-select'></div>
                        <span>Last 7 Days</span>
                    </div>

                    <div class='option' data-value='last-14-days'>
                        <div class='icon-select'></div>
                        <span>Last 14 Days</span>
                    </div>

                    <div class='option' data-value='last-30-days'>
                        <div class='icon-select'></div>
                        <span>Last 30 Days</span>
                    </div>
                </div>

                <div id="calendar-full-summary" class="calendar desktop active"></div>

                <div class='button-mobile'>
                    <div class='btn-save'>
                        <span>Save selection</span>

                        <div class='icon-check'></div>
                    </div>
                <div>
            </div>
        </div>
    `

    $('#report_page_content').find('.report-item.active .filters-menu').append($filter);

    createCalendarDays('calendar-full-summary', handleCalendarDaysAfterSelect);
    createCalendarDays('calendar-full-summary-mobile', handleCalendarDaysAfterSelect);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter.summary-period-selector')
        .on('click', handleSummaryPeriodSelectorClick);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter.summary-period-selector .option')
        .on('click', handleSummaryPeriodSelectorOptionClick);
}

function loadPeriodSelector(type) {
    const today = moment().format('YYYY-MM-DD');
    const firstDay = moment().startOf('month').format('YYYY-MM-DD');
    const lastDay = moment().endOf('month').format('YYYY-MM-DD');

    $filter = `
        <div class='filter period-selector active daily'>
            <div class='active-option'data-name='period-selector' data-time='Daily' data-value='{"startDate":"${firstDay}","endDate":"${today}"}'>
                <div class='custom-buttons'>
                    <div class='highlight'></div>

                    <div class='btn-select daily active' data-time='Daily'>
                        <span>Day</span>
                    </div>

                    <div class='btn-select monthly' data-time='Monthly'>
                        <span>Month</span>
                    </div>
                </div>

                <div class='custom-select daily active' data-type='month-to-date' data-value='{"startDate":"${firstDay}","endDate":"${today}"}'>
                    <div class='title'>
                        <span>Month to date</span>
                    </div>

                    <div class='icons'>
                        <div class='icon-toggle'></div>
                    </div>
                </div>

                <div class='custom-select monthly' data-type='current-month' data-value='{"startDate":"${firstDay}","endDate":"${lastDay}"}'>
                    <div class='title'>
                        <span>Current Month</span>
                    </div>

                    <div class='icons'>
                        <div class='icon-toggle'></div>
                    </div>
                </div>
            </div>

            <div class='wrapper'>
                <div class='options daily select-single active'>
                    <div class='option active' data-value='month-to-date'>
                        <div class='icon-select'></div>
                        <span>Month to date</span>
                    </div>

                    <div class='option custom-date daily' data-value='custom-date'>
                        <div class='icon-select'></div>
                        <span>Custom Date</span>
                    </div>

                    <div id="calendar-full-${type}-mobile" class="calendar mobile daily"></div>

                    <div class='option' data-value='yesterday'>
                        <div class='icon-select'></div>
                        <span>Yesterday</span>
                    </div>

                    <div class='option' data-value='last-month'>
                        <div class='icon-select'></div>
                        <span>Last Month</span>
                    </div>

                    <div class='option' data-value='last-7-days'>
                        <div class='icon-select'></div>
                        <span>Last 7 Days</span>
                    </div>

                    <div class='option' data-value='last-14-days'>
                        <div class='icon-select'></div>
                        <span>Last 14 Days</span>
                    </div>

                    <div class='option' data-value='last-30-days'>
                        <div class='icon-select'></div>
                        <span>Last 30 Days</span>
                    </div>
                </div>

                <div id="calendar-full-${type}" class="calendar daily active desktop"></div>

                <div class='options monthly select-single'>
                    <div class='option active' data-value='current-month'>
                        <div class='icon-select'></div>
                        <span>Current Month</span>
                    </div>

                    <div class='option custom-date monthly' data-value='custom-month'>
                        <div class='icon-select'></div>
                        <span>Custom Month</span>
                    </div>

                    <div id="calendar-${type}-mobile" class="calendar mobile monthly"></div>

                    <div class='option' data-value='last-month'>
                        <div class='icon-select'></div>
                        <span>Last Month</span>
                    </div>

                    <div class='option' data-value='current-year'>
                        <div class='icon-select'></div>
                        <span>Current Year</span>
                    </div>

                    <div class='option' data-value='last-year'>
                        <div class='icon-select'></div>
                        <span>Last Year</span>
                    </div>
                </div>

                <div id="calendar-${type}" class="calendar desktop monthly"></div>

                <div class='button-mobile'>
                    <div class='btn-save'>
                        <span>Save selection</span>

                        <div class='icon-check'></div>
                    </div>
                <div>
            </div>
        </div>
    `

    $('#report_page_content').find('.report-item.active .filters-menu').append($filter);

    createCalendarDays(`calendar-full-${type}`, handleCalendarDaysAfterSelect);
    createCalendarDays(`calendar-full-${type}-mobile`, handleCalendarDaysAfterSelect);

    createCalendarMonths(`calendar-${type}`, handleCalendarMonthsAfterSelect);
    createCalendarMonths(`calendar-${type}-mobile`, handleCalendarMonthsAfterSelect);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter.period-selector .custom-select')
        .on('click', handlePeriodSelectorClick);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter.period-selector .custom-buttons .btn-select')
        .on('click', handlePeriodSelectorButtonClick);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .button-mobile .btn-save')
        .off('click', handleSaveButtonClick)
        .on('click', handleSaveButtonClick);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter.period-selector .option')
        .on('click', handlePeriodSelectorOptionClick);
}

function loadChartSection() {
    $html = `
        <div class='report-page-chart'>
            <div class='title'>
                <h4>Daily Overview</h4>
            </div>

            <div class='chart-wrapper'>
                <div class='chart-container' style='position: relative;'>
                    <canvas id='report_canvas'></canvas>
                </div>
            </div>
        </div>
    `

    $('#report_page_content > .report-page-container > .report-page-main > .report-item.active > .bottom').append($html);
}

function loadTableSection(id, title) {
    $html = `
        <div class='report-page-table'>
            <div class='title'>
                <h4>${title}</h4>
            </div>

            <table class="table" id="${id}"></table>
        </div>
    `

    $('#report_page_content > .report-page-container > .report-page-main > .report-item.active > .bottom').append($html);
}

function loadError(error) {
    console.error('Error:', error);

    const errorDiv = $('#report_page_content').find('.report-item.active .report-page-error');
    if (errorDiv.length > 0) return;

    $('#report_page_content').find('.report-item.active').append(ERROR_HTML);
}



// @Dynamic Loading
function loadFilters(items, type) {
    let defaultTitle = '';
    let defaultClass = '';

    if (type === 'traffic_source') {
        defaultTitle = 'Traffic Source';
        defaultClass = 'traffic-source'
    }
    else if (type === 'advertiser') {
        defaultTitle = 'Advertiser';
        defaultClass = 'advertiser'
    }
    else if (type === 'affiliate_link') {
        defaultTitle = 'Affiliate Link';
        defaultClass = 'affiliate-link'
    }
    else if (type === 'sub_accounts') {
        defaultTitle = 'Sub Accounts';
        defaultClass = 'sub-accounts'
    }
    else return;

    $filter = `
        <div class='filter ${defaultClass}'>
            <div class='active-option' data-name='${defaultClass}' data-value='' data-default-title='${defaultTitle}'>
                <div class='title'>
                    <span>${defaultTitle}</span>
                </div>

                <div class='icons'>
                    <div class='icon-open'></div>
                    <div class='icon-clear'></div>
                </div>
            </div>
        
    `

    $filter += `
            <div class='wrapper'>
                <div class='options select-multiple'>
                    <div class='option' data-type='select-all' title='Select All'>
                        <div class='icon-select'></div>
                        <span>Select All</span>
                    </div>
    `

    items.map((item) => {
        $filter += `
            <div class='option' data-value='${item.id}' title='${item.name}'>
                <div class='icon-select'></div>
                <span>${item.name}</span>
            </div>
        `
    });

    $filter += `
                </div>

                <div class='button-mobile'>
                    <div class='btn-save'>
                        <span>Save selection</span>

                        <div class='icon-check'></div>
                    </div>

                    <div class='btn-clean'>
                        <span>Remove Selections</span>
                        
                        <div class='icon-trash'></div>
                    </div>
                <div>
            </div>
        </div>
    `

    $('#report_page_content').find('.report-item.active .filters-menu').append($filter);


    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter > .wrapper')
        .off('click', handleOptionsOutsideClick)
        .on('click', handleOptionsOutsideClick);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .button-mobile .btn-save')
        .off('click', handleSaveButtonClick)
        .on('click', handleSaveButtonClick);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .button-mobile .btn-clean')
        .off('click', handleCleanButtonClick)
        .on('click', handleCleanButtonClick);
}

function loadFiltersAction() {
    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter:not(.summary-period-selector, .period-selector, .network-earnings) .active-option > .title')
        .on('click', handleFilterTitleClick);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter:not(.summary-period-selector, .period-selector, .network-earnings) .active-option > .icons')
        .on('click', handleFilterIconClick);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter:not(.summary-period-selector, .period-selector, .network-earnings) .option')
        .on('click', handleFilterOptionClick);
}

function loadFiltersButtons() {
    $buttonApply = `
        <div class='btn-submit'>
            <span>Apply</span>

            <div class='icon-check'></div>
        </div>
    `

    $('#report_page_content').find('.report-item.active .filters-menu').append($buttonApply);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .btn-submit')
        .off('click', handleSubmitButtonClick)
        .on('click', handleSubmitButtonClick);
}

function loadFiltersEarnings(items, frequency) {
    const timeframeLabel = (frequency === 'monthly') ? 'Month' : 'Week';
    const timeframes = [];

    items.forEach((item, index) => {
        const [syear, smonth, sday] = item.start_date.split('-');
        const [eyear, emonth, eday] = item.end_date.split('-');

        const formattedStartDay = `${sday}/${smonth}/${syear}`;
        const formattedEndDay = `${eday}/${emonth}/${eyear}`;

        const counter = (frequency === 'monthly') ? items.length - index : moment(item.start_date, "YYYY-MM-DD").isoWeek(); // items.length - index

        let label;

        if (frequency === 'monthly') {
            label = `${getMonthLabelShort(parseInt(smonth, 10))} - ${syear}`; // (${timeframeLabel} ${counter} )
        }
        else {
            label = `${formattedStartDay} - ${formattedEndDay} (${timeframeLabel} ${counter})`;
        }

        timeframes.push({
            timeframe: item.timeframe,
            start: formattedStartDay,
            start_date: item.start_date,
            end: formattedEndDay,
            end_date: item.end_date,
            counter: counter,
            label: label
        });
    });

    const today = moment().format('YYYY-MM-DD');

    let startDate = '2000-01-01'
    let endDate = today;

    if (timeframes.length > 0) {
        startDate = timeframes[0].start_date;
        endDate = timeframes[timeframes.length - 1].end_date;
    }

    // items
    $filter = `
        <div class='filter earnings active ${frequency}'>
            <div class='active-option' title='${timeframes[0].label}' data-default-title='Period'>
                <div class='title'>
                    <span>${timeframes[0].label}</span>
                </div>

                <div class='icons'>
                    <div class='icon-toggle'></div>
                </div>
            </div>
    `

    $filter += `
            <div class='wrapper'>
                <div class='options select-multiple'>
                    <div class='option active' data-type='select-all' title='Select All'>
                        <div class='icon-select'></div>
                        <span>Select All</span>
                    </div>
    `

    timeframes.map((item) => {
        $filter += `
            <div class='option active' data-value='${item.timeframe}' title='${item.label}'>
                <div class='icon-select'></div>
                <span>${item.label}</span>
            </div>
        `
    });

    $filter += `
                </div>

                <div class='button-mobile'>
                    <div class='btn-save'>
                        <span>Save selection</span>

                        <div class='icon-check'></div>
                    </div>
                <div>
            </div>
        </div>
    `

    $('#report_page_content').find('.report-item.active .filters-menu').append($filter);

    // $('#report_page_content')
    //     .find('.report-item.active .filters-menu .button-mobile .btn-clean')
    //     .off('click', handleCleanButtonClick)
    //     .on('click', handleCleanButtonClick);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .button-mobile .btn-save')
        .off('click', handleSaveButtonClick)
        .on('click', handleSaveButtonClick);
}

function loadFiltersEarningsActions(records, filters) {
    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter.earnings')
        .on('click', handleEarningsClick);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter.earnings .option')
        .on('click', (e) => handleEarningsOptionClick(e, [{ title: 'earnings', items: records, table: 'Detailed View', key: 'earnings' }], filters));
}

function loadEarningsInfo(frequency) {
    const label = frequency.charAt(0).toUpperCase() + frequency.slice(1);

    $html = `
        <div class='info-earnings'>
            <div class='left'>
                <div class='icon'></div>
            </div>

            <div class='right'>
                <div class='info-label'>Your payment frequency:</div>
                <div class='info-frequency'>${label}</div>
            </div>
        </div>
    `

    $('#report_page_content').find('.report-item.active .filters-container').before($html);
}

function loadFiltersNetworkEarnings(items, frequency) {
    $filter = `
        <div class='filter network-earnings active'>
            <div class='active-option' title='${items[0].timeframe}' data-default-title='Period' data-name='network-earnings' data-value="{}" data-frequency='${frequency}'>
                <div class='title'>
                    <span>${items[0].timeframe}</span>
                </div>

                <div class='icons'>
                    <div class='icon-toggle'></div>
                </div>
            </div>
    `

    $filter += `
            <div class='wrapper'>
                <div class='options select-multiple'>
                    <div class='option active' data-type='select-all' title='Select All'>
                        <div class='icon-select'></div>
                        <span>Select All</span>
                    </div>
    `

    items.map((item) => {
        $filter += `
            <div class='option active' data-value='${item.timeframe}' title='${item.timeframe}'>
                <div class='icon-select'></div>
                <span>${item.timeframe}</span>
            </div>
        `
    });

    $filter += `
                </div>

                <div class='button-mobile'>
                    <div class='btn-save'>
                        <span>Save selection</span>

                        <div class='icon-check'></div>
                    </div>
                <div>
            </div>
        </div>
    `

    $('#report_page_content').find('.report-item.active .filters-menu').append($filter);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .button-mobile .btn-save')
        .off('click', handleSaveButtonClick)
        .on('click', handleSaveButtonClick);
}

function loadFiltersNetworkEarningsActions(recordsER, recordsAR, filters) {
    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter.network-earnings')
        .on('click', handleEarningsClick);

    $('#report_page_content')
        .find('.report-item.active .filters-menu .filter.network-earnings .option')
        .on('click', (e) => handleEarningsOptionClick(
            e,
            [
                { title: 'masterAffiliateNetworkER', items: recordsER, table: 'Earnings per Sub-Affiliate and Period', key: 'networkEarnings' },
                { title: 'masterAffiliateNetworkAR', items: recordsAR, table: 'Earnings per Sub-Affiliate, Period and Advertiser' }
            ],
            filters
        ));
}


// Key Metrics
function loadKeyMetrics(type, records) {
    $initial_html = `
        <div class='report-page-key-metrics'>
            <div class='title'>
                <h4>Key Metrics</h4>
            </div>

            <div class='cards'></div>
        </div>
    `;

    $('#report_page_content').find('.report-item.active > .bottom').append($initial_html);

    const metrics = calculateMetrics(records);
    const currency = records[0]?.currency ?? '';

    if (type === 'summary') loadKeyMetricsSummary(metrics, currency);
    else if (type === 'earnings') loadKeyMetricsEarnings(metrics, currency);
    else if (type === 'networkEarnings') loadKeyMetricsNetworkEarnings(metrics, currency);
}

function loadKeyMetricsSummary(metrics, currency) {
    if (metrics.ngr != null) {
        const totalNGR = formatDecimalNumber(metrics.ngr);
        loadKeyMetricsCard(`${currency} ${totalNGR}`, METRICS_INFO.totalNGR);
    }
    if (metrics.deposits != null) {
        const totalDeposits = formatDecimalNumber(metrics.deposits);
        loadKeyMetricsCard(`${currency} ${totalDeposits}`, METRICS_INFO.totalDeposits);
    }
    if (metrics.totalCommissions != null) {
        const totalCommissions = formatDecimalNumber(metrics.totalCommissions);
        loadKeyMetricsCard(`${currency} ${totalCommissions}`, METRICS_INFO.totalCommissions);
    }
    if (metrics.clicks != null) loadKeyMetricsCard(metrics.clicks, METRICS_INFO.clicks);
    if (metrics.signUps != null) loadKeyMetricsCard(metrics.signUps, METRICS_INFO.signUps);
    if (metrics.ftds != null) loadKeyMetricsCard(metrics.ftds, METRICS_INFO.ftds);
    if (metrics.cpas != null) loadKeyMetricsCard(metrics.cpas, METRICS_INFO.cpas);
}

function loadKeyMetricsEarnings(metrics, currency) {
    if (metrics.cpaCommissions != null) {
        const cpaCommissions = formatDecimalNumber(metrics.cpaCommissions);
        loadKeyMetricsCard(`${currency} ${cpaCommissions}`, METRICS_INFO.cpaCommissions);
    }
    if (metrics.rsCommissions != null) {
        const rsCommissions = formatDecimalNumber(metrics.rsCommissions);
        loadKeyMetricsCard(`${currency} ${rsCommissions}`, METRICS_INFO.rsCommissions);
    }
    if (metrics.totalCommissions != null) {
        const totalCommissions = formatDecimalNumber(metrics.totalCommissions);
        loadKeyMetricsCard(`${currency} ${totalCommissions}`, METRICS_INFO.totalCommissions);
    }
}
function loadKeyMetricsNetworkEarnings(metrics, currency) {
    if (metrics.cpaCommissions != null) {
        const cpaCommissions = formatDecimalNumber(metrics.cpaCommissions);
        loadKeyMetricsCard(`${currency} ${cpaCommissions}`, METRICS_INFO.cpaCommissions);
    }
    if (metrics.rsCommissions != null) {
        const rsCommissions = formatDecimalNumber(metrics.rsCommissions);
        loadKeyMetricsCard(`${currency} ${rsCommissions}`, METRICS_INFO.rsCommissions);
    }
    if (metrics.totalCommissions != null) {
        const totalCommissions = formatDecimalNumber(metrics.totalCommissions);
        loadKeyMetricsCard(`${currency} ${totalCommissions}`, METRICS_INFO.totalCommissions);
    }
}

function loadKeyMetricsCard(value, metric) {
    $card_html = `
        <div class='card'>
            <div class='top'>
                <div class='value'>
                    <span>${value}</span>
                </div>
            </div>
            <div class='bottom custom-tooltip'>
                <div class='title'>
                    <span>${metric.title}</span>
                </div>

                <div class='info-tooltip' data-title='${metric.info}'></div>
            </div>
        </div>
    `;

    $('#report_page_content').find('.report-item.active > .bottom .cards').append($card_html);
}


// Tables
function loadTable(type, formattedRecords, customTitle = 'Detailed View', time = 'Daily', paymentFrequency = null) {
    const id = `${type}Table`;
    const order = (type === 'earnings') ? [[0, 'desc']] : [];
    const frequency = paymentFrequency ? capitalizeFirst(paymentFrequency) : '';

    let columns = ['day', 'clicks', 'signUps', 'ftds', 'cpas', 'deposits', 'ngr', 'totalComissions'];
    if (time === 'Monthly') columns[0] = 'month';

    if (type === 'advertiser') columns.unshift('advertiser');
    else if (type === 'trafficSource') columns.unshift('trafficSource');
    else if (type === 'earnings') columns = ['period', 'cpaCommissions', 'rsCommissions', 'totalComissions'];
    else if (type === 'masterAffiliateTR') columns = ['subAccount', 'kycStatus', 'clicks', 'signUps', 'ftds', 'cpas', 'deposits', 'ngr', 'totalComissions'];
    else if (type === 'masterAffiliateAR') columns = ['subAccount', 'advertiser', 'clicks', 'signUps', 'ftds', 'cpas', 'deposits', 'ngr', 'totalComissions'];
    else if (type === 'masterAffiliateNetworkER') columns = ['subAccount', 'paymentFrequency', 'periodEarnings', 'cpaCommissions', 'rsCommissions', 'totalComissions'];
    else if (type === 'masterAffiliateNetworkAR') columns = ['subAccount', 'periodEarnings', 'advertiser', 'cpaCommissions', 'rsCommissions', 'totalComissions'];

    let sumCols = [1, 2, 3, 4, 5, 6, 7];

    if (type === 'advertiser' || type === 'trafficSource' || type === 'masterAffiliateTR' || type === 'masterAffiliateAR') sumCols = [2, 3, 4, 5, 6, 7, 8];
    else if (type === 'earnings') sumCols = [1, 2, 3];
    else if (type === 'masterAffiliateNetworkER' || type === 'masterAffiliateNetworkAR') sumCols = [3, 4, 5];

    let enableFooterCallback = true;
    if (type === 'summary' || type === 'earnings') enableFooterCallback = false;

    loadTableSection(id, customTitle);
    loadTableLayout(type, columns, formattedRecords, enableFooterCallback, frequency);
    loadTableData(id, order, sumCols, enableFooterCallback);
}

function loadTableLayout(type, labels, data, enableFooterCallback, frequency) {
    const table = $(`#${type}Table`);

    let html = '<thead><tr>';

    labels.forEach((label, index) => {
        const tableInfo = TABLE_INFO[label];

        html += `
            <th>
                <span class='title custom-tooltip'>
                    ${tableInfo.title}

                    ${tableInfo.info ? `<span data-title='${tableInfo.info}' class='info-tooltip' ${labels.length === index + 1 ? "tooltip-direction='left'" : ''}></span>` : ''}
                </span>
            </th>
        `
    });

    html += '</tr></thead><tbody>';

    if (type === 'advertiser') html += loadTableRowsAdvertiser(data);
    else if (type === 'trafficSource') html += loadTableRowsTrafficSource(data);
    else if (type === 'earnings') html += loadTableRowsEarnings(data);
    else if (type === 'masterAffiliateTR') html += loadTableRowsMasterAffiliateTR(data);
    else if (type === 'masterAffiliateAR') html += loadTableRowsMasterAffiliateAR(data);
    else if (type === 'masterAffiliateNetworkER') html += loadTableRowsMasterAffiliateNetworkER(data, frequency);
    else if (type === 'masterAffiliateNetworkAR') html += loadTableRowsMasterAffiliateNetworkAR(data);
    else html += loadTableRows(data);

    html += '</tbody>';

    if (enableFooterCallback) {
        html += `
            <tfoot>
                <tr>
                    <th>
                        <span class='title custom-tooltip'>
                            Total
                            <span data-title='Total numbers for the selected dates.' class='info-tooltip' tooltip-direction='top'></span>
                        </span>
                    </th>
        `;

        for (i = 0; i < labels.length - 1; i++) {
            html += '<th></th>';
        }

        html += '</tr></tfoot>';
    }

    table.html(html);
}

function loadTableRows(data) {
    let $rows = '';

    data.forEach((item) => {
        let label, order = '';

        if (item.day) {
            label = item.day;
            order = formatDate(item.day, 'YYYY-MM-DD');
        }
        else if (item.month) {
            const [year, month] = item.month.split('-');
            label = `${getMonthLabelShort(parseInt(month, 10))} - ${year}`;
            order = item.month;
        }

        $rows += `
            <tr>
                <td data-order="${order}">
                    ${label}
                </td>
                <td>${item.clicks}</td>
                <td>${item.signUps}</td>
                <td>${item.ftds}</td>
                <td>${item.cpas}</td>
                <td class='${item.deposits < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.deposits}</span>
                </td>
                <td class='${item.ngr < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.ngr}</span>
                </td>
                <td class='${item.totalCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.totalCommissions}</span>
                </td>
            </tr>
         `
    });

    return $rows;
}

function loadTableRowsAdvertiser(data) {
    let $rows = '';

    data.forEach((item) => {
        let label, order = '';

        if (item.day) {
            label = item.day;
            order = formatDate(item.day, 'YYYY-MM-DD');
        }
        else if (item.month) {
            const [year, month] = item.month.split('-');
            label = `${getMonthLabelShort(parseInt(month, 10))} - ${year}`;
            order = item.month;
        }

        $rows += `
            <tr>
                <td title='${item.advertiser.name}'>
                    <img src='${item.advertiser.logoUrl}' alt='${item.advertiser.name}' width='80'>
                </td>
                <td data-order="${order}">
                    ${label}
                </td>
                <td>${item.clicks}</td>
                <td>${item.signUps}</td>
                <td>${item.ftds}</td>
                <td>${item.cpas}</td>
                <td class='${item.deposits < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.deposits}</span>
                </td>
                <td class='${item.ngr < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.ngr}</span>
                </td>
                <td class='${item.totalCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.totalCommissions}</span>
                </td>
            </tr>
         `
    });

    return $rows;
}

function loadTableRowsTrafficSource(data) {
    let $rows = '';

    data.forEach((item) => {
        let label, order = '';

        if (item.day) {
            label = item.day;
            order = formatDate(item.day, 'YYYY-MM-DD');
        }
        else if (item.month) {
            const [year, month] = item.month.split('-');
            label = `${getMonthLabelShort(parseInt(month, 10))} - ${year}`;
            order = item.month;
        }

        $rows += `
            <tr>
                <td title='${item.trafficSource.name}'>
                    <a href='${item.trafficSource.url}' target='_blank'>${item.trafficSource.name}</a>
                </td>
                <td data-order="${order}">
                    ${label}
                </td>
                <td>${item.clicks}</td>
                <td>${item.signUps}</td>
                <td>${item.ftds}</td>
                <td>${item.cpas}</td>
                <td class='${item.deposits < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.deposits}</span>
                </td>
                <td class='${item.ngr < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.ngr}</span>
                </td>
                <td class='${item.totalCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.totalCommissions}</span>
                </td>
            </tr>
         `
    });

    return $rows;
}

function loadTableRowsEarnings(data) {
    let $rows = '';

    data.forEach((item) => {
        let label;

        if (item.frequency === 'monthly') {
            label = `${getMonthLabelShort(parseInt(item.smonth, 10))} - ${item.syear}`;
        }
        else {
            label = `${item.start} - ${item.end} ${item.label}`
        }

        // <td data-order="${formatDate(item.start, 'YYYY-MM-DD')}">
        $rows += `
            <tr data-period='${item.period}'>
                <td data-order="${item.period}">
                    ${label}
                </td>
                <td class='${item.cpaCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.cpaCommissions}</span>
                </td>
                <td class='${item.rsCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.rsCommissions}</span>
                </td>
                <td class='${item.totalCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.totalCommissions}</span>
                </td>
            </tr>
         `
    });

    return $rows;
}

function loadTableRowsMasterAffiliateTR(data) {
    let $rows = '';

    data.forEach((item) => {
        $rows += `
            <tr>
                <td>${item.subAccount}</td>
                <td class='kyc-status'>
                    <span class='${STATUS_MAP[item.kycStatus]}'>${item.kycStatus}</span>
                </td>
                <td>${item.clicks}</td>
                <td>${item.signUps}</td>
                <td>${item.ftds}</td>
                <td>${item.cpas}</td>
                <td class='${item.deposits < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.deposits}</span>
                </td>
                <td class='${item.ngr < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.ngr}</span>
                </td>
                <td class='${item.totalCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.totalCommissions}</span>
                </td>
            </tr>
         `
    });

    return $rows;
}

function loadTableRowsMasterAffiliateAR(data) {
    let $rows = '';

    data.forEach((item) => {
        $rows += `
            <tr>
                <td>${item.subAccount}</td>
                <td title='${item.advertiser.name}'>
                    <img src='${item.advertiser.logoUrl}' alt='${item.advertiser.name}' width='80'>
                </td>
                <td>${item.clicks}</td>
                <td>${item.signUps}</td>
                <td>${item.ftds}</td>
                <td>${item.cpas}</td>
                <td class='${item.deposits < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.deposits}</span>
                </td>
                <td class='${item.ngr < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.ngr}</span>
                </td>
                <td class='${item.totalCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.totalCommissions}</span>
                </td>
            </tr>
         `
    });

    return $rows;
}

function loadTableRowsMasterAffiliateNetworkER(data, frequency) {
    let $rows = '';

    data.forEach((item) => {
        $rows += `
            <tr>
                <td>${item.subAccount}</td>
                <td>${frequency}</td>
                <td data-order="${item.period}">
                    ${item.period}
                </td>
                <td class='${item.cpaCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.cpaCommissions}</span>
                </td>
                <td class='${item.rsCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.rsCommissions}</span>
                </td>
                <td class='${item.totalCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.totalCommissions}</span>
                </td>
            </tr>
         `
    });

    return $rows;
}

function loadTableRowsMasterAffiliateNetworkAR(data) {
    let $rows = '';

    data.forEach((item) => {
        $rows += `
            <tr>
                <td>${item.subAccount}</td>
                <td data-order="${item.period}">
                    ${item.period}
                </td>
                <td title='${item.advertiser.name}'>
                    <img src='${item.advertiser.logoUrl}' alt='${item.advertiser.name}' width='80'>
                </td>
                <td class='${item.cpaCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.cpaCommissions}</span>
                </td>
                <td class='${item.rsCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.rsCommissions}</span>
                </td>
                <td class='${item.totalCommissions < 0 ? 'negative' : ''}'>
                    <span class='currency'>${item.currency ?? ''}</span>
                    <span>${item.totalCommissions}</span>
                </td>
            </tr>
         `
    });

    return $rows;
}

function loadTableData(tableId, order = [], sumCols, enableFooterCallback) {
    const options = {
        dom: "<'row'<'col-sm-12'<'table-scrollable' <'table-scrollable table-wrapper' tr >>>>" +
            "<'row listing-split-page'<'col-sm-5'i><'col-sm-7'p>>",
        language: {
            paginate: {
                previous: "<i class=\"fa fa-angle-left\"></i>",
                next: "<i class=\"fa fa-angle-right\"></i>"
            }
        },
        order: order
    }

    if (enableFooterCallback) {
        options.footerCallback = function (row, data, start, end, display) {
            const api = this.api();

            sumCols.forEach((colIdx) => {
                const columnData = api.column(colIdx).data().toArray();

                const currency = detectCurrency(columnData);

                const total = columnData.reduce((acc, v) => acc + parseNumber(v), 0);

                const formatted = currency
                    ? `${currency} ${total.toFixed(2)}`
                    : total.toString();

                $(api.column(colIdx).footer()).html(formatted);
            });
        }
    }

    $(`#${tableId}`).DataTable(options);
}


// @Aux Other
function calculateMetrics(records) {
    const metrics = {};

    records.forEach(record => {
        Object.entries(record).forEach(([key, value]) => {
            if (!isNaN(value) && value !== null && value !== "") {
                metrics[key] = (metrics[key] || 0) + Number(value);
            }
        });
    });

    return metrics;
}

function formatDate(dateString, format = 'DD/MM/YYYY') {
    let year, month, day;

    if (dateString.includes('-')) [year, month, day] = dateString.split('-'); // format YYYY-MM-DD
    else if (dateString.includes('/')) [day, month, year] = dateString.split('/'); // format DD/MM/YYYY
    else return dateString;

    if (format === 'DD/MM/YYYY') return `${day}/${month}/${year}`
    else if (format === 'YYYY-MM-DD') return `${year}-${month}-${day}`

    return dateString;
}

function formatDecimalNumber(number, digits = 2) {
    return number.toLocaleString('pt-BR', { minimumFractionDigits: digits, maximumFractionDigits: digits });
}

function formatRecords(records, extra = null) {
    const formattedRecords = [];

    records.forEach((record) => {
        const formattedDate = record.Day ? formatDate(record.Day) : null;

        const row = {
            day: formattedDate,
            month: record.Month ? record.Month : null,
            clicks: record.NumberOfClicks,
            signUps: record.NumberOfSignups,
            ftds: record.NumberOfFTDs,
            cpas: record.NumberOfCPAs,
            deposits: record.Deposits,
            ngr: record.NGR,
            totalCommissions: record.TotalCommissions,
            currency: record.Currency ? record.Currency : ''
        }

        if (extra === 'advertiser') {
            row.advertiser = {
                name: record.Advertiser ? record.Advertiser : '',
                logoUrl: record.AdvertiserLogoUrl ? record.AdvertiserLogoUrl : ''
            }
        }
        else if (extra === 'traffic_source') {
            row.trafficSource = {
                name: record.TrafficSource ? record.TrafficSource : '',
                url: record.TrafficSourceUrl ? record.TrafficSourceUrl : ''
            }
        }

        formattedRecords.push(row);
    });

    return formattedRecords;
}

function formatRecordsEarnings(records, filters) {
    const paymentFrequency = filters.payment_frequency;
    const formattedTimeframes = formatTimeframes(filters.timeframes);

    const timeframeLabel = (paymentFrequency === 'monthly') ? 'Month' : 'Week';
    const formattedRecords = [];

    records.forEach((item, index) => {
        const timeframe = formattedTimeframes[item.Period];

        // if (!timeframe) console.log(`#${index}: No Timeframe: ${item.Period}`);
        // else if (!timeframe.start_date) console.log(timeframe);

        if (timeframe) {
            const counter = (paymentFrequency === 'monthly') ? timeframe.counter : moment(timeframe.start_date, "YYYY-MM-DD").isoWeek();
            const [syear, smonth] = item.Period.split('-');

            formattedRecords.push({
                start: timeframe.start,
                end: timeframe.end,
                counter: counter,
                label: `(${timeframeLabel} ${counter})`,
                frequency: paymentFrequency,
                syear,
                smonth,
                period: item.Period,
                cpaCommissions: item.CPACommissions,
                rsCommissions: item.RSCommissions,
                totalCommissions: item.TotalCommissions,
                currency: item.Currency ? item.Currency : ''
            });
        }
    });

    return formattedRecords;
}

function formatRecordsMasterAffiliate(records, extra = null) {
    const formattedRecords = [];

    records.forEach((record) => {
        const row = {
            subAccount: record.SubAccount,
            clicks: record.NumberOfClicks,
            signUps: record.NumberOfSignups,
            ftds: record.NumberOfFTDs,
            cpas: record.NumberOfCPAs,
            deposits: record.Deposits,
            ngr: record.NGR,
            totalCommissions: record.TotalCommissions,
            currency: record.Currency ? record.Currency : '€'
        }

        if (extra === 'advertiser') {
            row.advertiser = {
                name: record.Advertiser ? record.Advertiser : '',
                logoUrl: record.AdvertiserLogoUrl ? record.AdvertiserLogoUrl : ''
            }
        }
        else if (extra === 'kycStatus') {
            row.kycStatus = record.SubAccountKYCStatus;
        }

        formattedRecords.push(row);
    });

    return formattedRecords;
}

function formatRecordsMasterAffiliateNetwork(records, extra = null) {
    const formattedRecords = [];

    records.forEach((record) => {
        const row = {
            period: record.Period,
            subAccount: record.SubAccount,
            cpaCommissions: record.CPACommissions,
            rsCommissions: record.RSCommissions,
            totalCommissions: record.TotalCommissions,
            currency: record.Currency ? record.Currency : '€'
        }

        if (extra === 'advertiser') {
            row.advertiser = {
                name: record.Advertiser ? record.Advertiser : '',
                logoUrl: record.AdvertiserLogoUrl ? record.AdvertiserLogoUrl : ''
            }
        }

        formattedRecords.push(row);
    });

    return formattedRecords;
}

function formatTimeframes(timeframes) {
    const formattedTimeframes = {};

    timeframes.forEach((item, index) => {
        const [syear, smonth, sday] = item.start_date.split('-');
        const [eyear, emonth, eday] = item.end_date.split('-');

        const formattedStartDay = `${sday}/${smonth}/${syear}`;
        const formattedEndDay = `${eday}/${emonth}/${eyear}`;

        formattedTimeframes[item.timeframe] = {
            start_date: item.start_date,
            end_date: item.end_date,
            start: formattedStartDay,
            end: formattedEndDay,
            counter: timeframes.length - index
        };
    });

    return formattedTimeframes;
}

function parseNumber(n) {
    if (typeof n === 'number') return n;
    if (typeof n === 'string') {
        const cleaned = n.replace(/[^\d.,-]/g, '').replace(/\s+/g, ''); // Regex to clean digits leaving only number + decimals symbols

        const normalized = cleaned.includes(',') && !cleaned.includes('.') // Normalize number to use . for decimals
            ? cleaned.replace(/\./g, '').replace(',', '.')
            : cleaned.replace(/,/g, '');

        const num = parseFloat(normalized);

        return isNaN(num) ? 0 : num;
    }
    return 0;
}

function detectCurrency(string) {
    for (const char of string) {
        if (typeof char !== 'string') continue;

        const currency = char.match(/\b(R\$)|[€$£]/);

        if (currency) return currency[0];
    }

    return null;
}

function capitalizeFirst(str) {
    if (!str) return '';
    return str.charAt(0).toUpperCase() + str.slice(1);
}



// @Aux Handlers
function handleFilterTitleClick(e) {
    const filter = $(e.currentTarget).closest('.filter')[0];

    $(filter).toggleClass('open');
}

function handleFilterIconClick(e) {
    const filter = $(e.currentTarget).closest('.filter')[0];

    if ($(filter).hasClass('open')) {
        if ($(filter).hasClass('active')) {
            const activeOption = $(filter).find('.active-option')[0];

            activeOption.dataset.value = '';
            $(activeOption).attr('title', '');
            $(activeOption).find('span').text(activeOption.dataset.defaultTitle);

            $(filter).find('.option').removeClass('active');
            $(filter).removeClass('active');
        }

        $(filter).removeClass('open');

        return;
    }

    $(filter).addClass('open');
}

function handleFilterOptionClick(e) {
    e.stopPropagation();

    const option = $(e.currentTarget)[0];
    const filter = $(option).closest('.filter');

    $(option).toggleClass('active');

    if (option.dataset.type === 'select-all') {
        const options = $(option).closest('.options').find('.option').not(option);

        if ($(option).hasClass('active')) {
            $(options).addClass('active');
        }
        else {
            $(options).removeClass('active');
        }
    }

    const allSelectedOptions = $(filter).closest('.filters-menu').find('.filter.active');
    const filtersContainer = $(filter).closest('.filters-container').find('.filters-mobile span.counter');

    let filtersCounter = allSelectedOptions.length <= 1 ? `${allSelectedOptions.length} filter applied` : `${allSelectedOptions.length} filters applied`;
    $(filtersContainer).text(filtersCounter);

    const selectedOptions = $(filter).find('.option.active');
    const activeOption = $(filter).find('.active-option')[0];

    if (selectedOptions.length === 0) {
        filter.removeClass('active');

        activeOption.dataset.value = '';
        $(activeOption).attr('title', '');
        $(activeOption).find('span').text(activeOption.dataset.defaultTitle);

        return;
    }

    if (!filter.hasClass('active')) filter.addClass('active');

    const values = [];
    const labels = [];

    selectedOptions.each((index, item) => {
        if (item.dataset.type === 'select-all') return;

        values.push(item.dataset.value);
        labels.push($(item).find('span').text());
    });

    activeOption.dataset.value = JSON.stringify(values);

    const firstLabel = labels[0];
    const remaining = labels.length - 1;

    const displayText = remaining > 0 ? `${firstLabel} +${remaining}` : firstLabel;

    $(activeOption).find('span').text(displayText);
    $(activeOption).attr('title', labels.join(', '));
}

function handleSummaryPeriodSelectorClick(e) {
    const filter = $(e.currentTarget)[0];

    $(filter).toggleClass('open');
}

function handleSummaryPeriodSelectorOptionClick(e) {
    e.stopPropagation();

    const option = $(e.currentTarget)[0];
    const filter = $(option).closest('.filter.summary-period-selector');

    if (option.dataset.value !== 'custom-date') {
        if ($(option).hasClass('active')) {
            $(filter).removeClass('open');
            return;
        }

        $(filter).find('.option').not(option).removeClass('active');
        $(option).toggleClass('active');

        handleDateRangeSelector(option.dataset.value, option, filter);
        $(filter).removeClass('open');
    }
    else {
        $(filter).find('.option').not(option).removeClass('active');
        if (!$(option).hasClass('active')) $(option).addClass('active');
    }
}

function getDateRangeInterval(interval) {
    const today = moment().format('YYYY-MM-DD');
    const firstDay = moment().startOf('month').format('YYYY-MM-DD');
    const lastDay = moment().endOf('month').format('YYYY-MM-DD');

    let startDate, endDate;

    if (interval === 'month-to-date') {
        startDate = firstDay;
        endDate = today;
    }
    else if (interval === 'yesterday') {
        const yesterday = moment().subtract(1, 'day').format('YYYY-MM-DD');

        startDate = yesterday;
        endDate = yesterday;
    }
    else if (interval === 'last-month') {
        const firstDayLastMonth = moment().subtract(1, 'month').startOf('month').format('YYYY-MM-DD');
        const lastDayLastMonth = moment().subtract(1, 'month').endOf('month').format('YYYY-MM-DD');

        startDate = firstDayLastMonth;
        endDate = lastDayLastMonth;
    }
    else if (interval === 'last-7-days') {
        const startLast7Days = moment().subtract(6, 'days').format('YYYY-MM-DD');

        startDate = startLast7Days;
        endDate = today;
    }
    else if (interval === 'last-14-days') {
        const startLast14Days = moment().subtract(13, 'days').format('YYYY-MM-DD');

        startDate = startLast14Days;
        endDate = today;
    }
    else if (interval === 'last-30-days') {
        const startLast30Days = moment().subtract(29, 'days').format('YYYY-MM-DD');

        startDate = startLast30Days;
        endDate = today;
    }
    else if (interval === 'current-month') {
        startDate = firstDay;
        endDate = lastDay;
    }
    else if (interval === 'current-year') {
        const firstDayYear = moment().startOf('year').format('YYYY-MM-DD');
        const lastDayYear = moment().endOf('year').format('YYYY-MM-DD');

        startDate = firstDayYear;
        endDate = lastDayYear;
    }
    else if (interval === 'last-year') {
        const firstDayLastYear = moment().subtract(1, 'year').startOf('year').format('YYYY-MM-DD');
        const lastDayLastYear = moment().subtract(1, 'year').endOf('year').format('YYYY-MM-DD');

        startDate = firstDayLastYear;
        endDate = lastDayLastYear;
    }

    return { startDate, endDate }
}

function handleDateRangeSelector(interval, option, filter) {
    const dateRange = getDateRangeInterval(interval);

    const activeOption = $(filter).find('.active-option')[0];
    const label = $(option).find('span').text();

    activeOption.dataset.type = interval;
    activeOption.dataset.value = JSON.stringify(dateRange);

    const customSelect = $(filter).find('.custom-select.active')[0];

    if (customSelect) {
        customSelect.dataset.type = interval;
        customSelect.dataset.value = JSON.stringify(dateRange);

        $(customSelect).find('span').text(label);
    }
    else {
        $(activeOption).find('span').text(label);
    }
}

function handlePeriodSelectorOptionClick(e) {
    e.stopPropagation();

    const option = $(e.currentTarget)[0];
    const filter = $(option).closest('.filter.period-selector');

    if (option.dataset.value !== 'custom-date' && option.dataset.value !== 'custom-month') {
        if ($(option).hasClass('active')) {
            $(filter).removeClass('open');
            return;
        }

        $(filter).find('.options.active .option').not(option).removeClass('active');
        $(option).toggleClass('active');

        handleDateRangeSelector(option.dataset.value, option, filter);
        $(filter).removeClass('open');
    }
    else {
        $(filter).find('.options.active .option').not(option).removeClass('active');
        if (!$(option).hasClass('active')) $(option).addClass('active');
    }
}

function handlePeriodSelectorClick(e) {
    const select = e.currentTarget;
    const filter = $(select).closest('.filter')[0];

    $(filter).toggleClass('open');
}

function handlePeriodSelectorButtonClick(e) {
    const button = e.currentTarget;
    if ($(button).hasClass('active')) return;

    const filter = $(button).closest('.filter')[0];

    if ($(filter).hasClass('daily')) $(filter).removeClass('daily').addClass('monthly');
    else $(filter).removeClass('monthly').addClass('daily');

    $(filter).find('.custom-select').toggleClass('active');
    $(filter).find('.custom-buttons .btn-select').toggleClass('active');
    $(filter).find('.options').toggleClass('active');
    $(filter).find('.calendar.desktop').toggleClass('active');

    const activeOption = $(filter).find('.active-option')[0];
    const customSelect = $(filter).find('.custom-select.active')[0];

    activeOption.dataset.value = customSelect.dataset.value;
    activeOption.dataset.time = button.dataset.time;
}

function handleSubmitButtonClick(e) {
    e.stopPropagation();

    const activeReport = $('#report_page_content').find('.report-item.active');
    const activeOptions = $(activeReport).find('.filters-menu .filter.active .active-option');

    let time = 'Daily';
    let frequency = 'Weekly';

    const filters = {}

    activeOptions.each((index, filter) => {
        const name = filter.dataset.name;
        const value = JSON.parse(filter.dataset.value);

        if (name === 'period-selector') {
            time = filter.dataset.time;
            filters.startDate = value.startDate;
            filters.endDate = value.endDate;
        }
        else if (name === 'traffic-source') {
            filters.trafficSourceRukoId = value;
        }
        else if (name === 'advertiser') {
            filters.advertiserRukoId = value;
        }
        else if (name === 'affiliate-link') {
            filters.affiliateLinkRukoId = value;
        }
        else if (name === 'sub-accounts') {
            filters.sapSubAccountRukoId = value;
        }
        else if (name === 'network-earnings') {
            frequency = filter.dataset.frequency;
        }
    });

    $(activeReport).find('.filters-menu .filter').removeClass('open'); // Close any open options

    removeSections(activeReport);
    loadFiltersData($(activeReport).attr('id'), time, filters, frequency);
}

function handleEarningsClick(e) {
    const filter = $(e.currentTarget)[0];

    $(filter).toggleClass('open');
}

function handleEarningsOptionClick(e, records, filters) {
    e.stopPropagation();

    const activeReport = $('#report_page_content').find('.report-item.active');

    const option = $(e.currentTarget)[0];
    const activeOption = $(option).closest('.filter').find('.active-option');

    $(option).toggleClass('active');

    if (option.dataset.type === 'select-all') {
        const options = $(option).closest('.options').find('.option').not(option);

        if ($(option).hasClass('active')) {
            $(options).addClass('active');
        }
        else {
            $(options).removeClass('active');
        }
    }

    const allOptios = $(option).parent().find('.option.active');

    if (allOptios.length > 0) {
        const firstOption = allOptios[0].dataset.type === 'select-all' ? allOptios[1] : allOptios[0];
        const newTitle = $(firstOption).attr('title');

        $(activeOption).attr('title', newTitle);
        $(activeOption).find('.title span').text(newTitle);
    }
    else {
        const newTitle = activeOption[0].dataset.defaultTitle

        $(activeOption).attr('title', newTitle);
        $(activeOption).find('.title span').text(newTitle);
    }

    const activeValues = $(option).parent().find('.option.active').map(function () {
        return $(this).data('value');
    }).get();


    removeSections(activeReport);

    records.forEach((record, index) => {
        const newRecords = record.items.filter(item => activeValues.includes(item.period));

        if (index === 0) loadKeyMetrics(record.key, newRecords);

        loadTable(record.title, newRecords, record.table, 'Daily', filters.payment_frequency);
    });
}

function handleFiltersMobileClick(e) {
    e.stopPropagation();

    const filtersMobileDiv = $(e.currentTarget)[0];
    const filtersDiv = $(filtersMobileDiv).parent().find('.report-item-filters');

    $(filtersMobileDiv).toggleClass('open');
    $(filtersDiv).toggleClass('open');
}

function handleOptionsOutsideClick(e) {
    e.stopPropagation();

    if (window.innerWidth > 1024) return;

    const target = e.target;

    const wrapper = $(e.currentTarget).closest('.filter')[0];
    if (!wrapper) return;

    const wasInsideWrapper = wrapper.contains(target);
    const allowedEl = target.closest('.options, .button-mobile');

    const isAllowed = !!(allowedEl && wrapper.contains(allowedEl));

    if (!wasInsideWrapper || (wasInsideWrapper && !isAllowed)) {
        $(wrapper).removeClass('open');
    }
}

function handleSaveButtonClick(e) {
    e.stopPropagation();

    const filter = $(e.currentTarget).closest('.filter')[0];

    $(filter).toggleClass('open');
}

function handleCleanButtonClick(e) {
    e.stopPropagation();

    const filter = $(e.currentTarget).closest('.filter')[0];

    if ($(filter).hasClass('open')) {
        if ($(filter).hasClass('active')) {
            const activeOption = $(filter).find('.active-option')[0];

            activeOption.dataset.value = '';
            $(activeOption).attr('title', '');
            $(activeOption).find('span').text(activeOption.dataset.defaultTitle);

            $(filter).find('.option').removeClass('active');
            $(filter).removeClass('active');
        }

        $(filter).removeClass('open');

        return;
    }

    $(filter).addClass('open');
}



// @Remove Section
function removeSections(activeReport) {
    $(activeReport).find('.report-page-key-metrics').remove();
    $(activeReport).find('.report-page-chart').remove();
    $(activeReport).find('.report-page-table').remove();
}



// @Ajax
function getAjaxSettingsRukoIdProxy(rukoUserId) {
    const endpoint = '/reports/MasterAffiliate/IsMasterAccount';
    const url = MIDDLEWARE_URL + endpoint;

    return {
        url: url,
        method: 'POST',
        timeout: 0,
        headers: { 'Content-Type': 'application/json' },
        data: JSON.stringify({
            sapAccountRukoId: rukoUserId
        }),
    };
}

function getAjaxSettingsFiltersProxy(rukoUserId, type) {
    const endpoint = '/reports/ReportFilters';
    const url = MIDDLEWARE_URL + endpoint;

    const settings = {
        url: url,
        method: 'POST',
        timeout: 0,
        headers: { 'Content-Type': 'application/json' },
        data: JSON.stringify({
            type: type,
            sapAccountRukoId: rukoUserId
        })
    };

    return settings;
}

function getAjaxSettingsProxy(endpointPath, rukoUserId, customFilters = null) {
    const url = MIDDLEWARE_URL + endpointPath;
    const filters = customFilters ?? {};

    if (!(filters.startDate && filters.endDate)) {
        const now = new Date();

        const todayUTC = new Date(Date.UTC(
            now.getUTCFullYear(),
            now.getUTCMonth(),
            now.getUTCDate()
        ));

        const firstDayUTC = new Date(Date.UTC(
            todayUTC.getUTCFullYear(),
            todayUTC.getUTCMonth(),
            1
        ));

        filters.startDate = firstDayUTC.toISOString().slice(0, 10); // "YYYY-MM-DD"
        filters.endDate = todayUTC.toISOString().slice(0, 10);     // "YYYY-MM-DD"
    }

    return {
        url: url,
        method: 'POST',
        timeout: 0,
        headers: { 'Content-Type': 'application/json' },
        data: JSON.stringify({
            filters,
            pagination: {},
            sapAccountRukoId: rukoUserId
        }),
        beforeSend: function () {
            $('#report_page_content').addClass('loading');
        }
    };
}



// @Fetch
function fetchIsMasterAccount(rukoUserId) {
    const settings = getAjaxSettingsRukoIdProxy(rukoUserId);

    return new Promise((resolve, reject) => {
        $.ajax(settings)
            .done((response) => resolve(response))
            .fail((jqXHR, textStatus, errorThrown) => {
                console.error('Request failed:', textStatus, errorThrown);
                console.error('Server response:', jqXHR.responseText);
                reject(errorThrown);
            });
    });
}

function fetchReportFilters(rukoUserId, type) {
    const settings = getAjaxSettingsFiltersProxy(rukoUserId, type);

    return new Promise((resolve, reject) => {
        $.ajax(settings)
            .done((response) => resolve(response))
            .fail((jqXHR, textStatus, errorThrown) => {
                console.error('Request failed:', textStatus, errorThrown);
                console.error('Server response:', jqXHR.responseText);
                reject(errorThrown);
            });
    });
}

function fetchSummaryReport(rukoUserId, filters = null) {
    const settings = getAjaxSettingsProxy('/reports/SummaryReport', rukoUserId, filters);

    return new Promise((resolve, reject) => {
        $.ajax(settings)
            .done((response) => resolve(response))
            .fail((jqXHR, textStatus, errorThrown) => {
                console.error('Request failed:', textStatus, errorThrown);
                console.error('Server response:', jqXHR.responseText);
                reject(errorThrown);
            });
    });
}

function fetchTrafficReport(rukoUserId, time, filters = null) {
    const settings = getAjaxSettingsProxy('/reports/TrafficReport/' + time, rukoUserId, filters);

    return new Promise((resolve, reject) => {
        $.ajax(settings)
            .done((response) => resolve(response))
            .fail((jqXHR, textStatus, errorThrown) => {
                console.error('Request failed:', textStatus, errorThrown);
                console.error('Server response:', jqXHR.responseText);
                reject(errorThrown);
            });
    });
}

function fetchAdvertiserReport(rukoUserId, time, filters = null) {
    const settings = getAjaxSettingsProxy('/reports/AdvertiserReport/' + time, rukoUserId, filters);

    return new Promise((resolve, reject) => {
        $.ajax(settings)
            .done((response) => resolve(response))
            .fail((jqXHR, textStatus, errorThrown) => {
                console.error('Request failed:', textStatus, errorThrown);
                console.error('Server response:', jqXHR.responseText);
                reject(errorThrown);
            });
    });
}

function fetchTrafficSourceReport(rukoUserId, time, filters = null) {
    const settings = getAjaxSettingsProxy('/reports/TrafficSourceReport/' + time, rukoUserId, filters);

    return new Promise((resolve, reject) => {
        $.ajax(settings)
            .done((response) => resolve(response))
            .fail((jqXHR, textStatus, errorThrown) => {
                console.error('Request failed:', textStatus, errorThrown);
                console.error('Server response:', jqXHR.responseText);
                reject(errorThrown);
            });
    });
}

function fetchEarningsReport(rukoUserId, filters = null) {
    const settings = getAjaxSettingsProxy('/reports/EarningsReport', rukoUserId, filters);

    return new Promise((resolve, reject) => {
        $.ajax(settings)
            .done((response) => resolve(response))
            .fail((jqXHR, textStatus, errorThrown) => {
                console.error('Request failed:', textStatus, errorThrown);
                console.error('Server response:', jqXHR.responseText);
                reject(errorThrown);
            });
    });
}

function fetchMasterAffiliateTrafficReport(rukoUserId, filters = null) {
    const settings = getAjaxSettingsProxy('/reports/MasterAffiliate/TrafficReport', rukoUserId, filters);

    return new Promise((resolve, reject) => {
        $.ajax(settings)
            .done((response) => resolve(response))
            .fail((jqXHR, textStatus, errorThrown) => {
                console.error('Request failed:', textStatus, errorThrown);
                console.error('Server response:', jqXHR.responseText);
                reject(errorThrown);
            });
    });
}

function fetchMasterAffiliateAdvertiserReport(rukoUserId, filters = null) {
    const settings = getAjaxSettingsProxy('/reports/MasterAffiliate/AdvertiserReport', rukoUserId, filters);

    return new Promise((resolve, reject) => {
        $.ajax(settings)
            .done((response) => resolve(response))
            .fail((jqXHR, textStatus, errorThrown) => {
                console.error('Request failed:', textStatus, errorThrown);
                console.error('Server response:', jqXHR.responseText);
                reject(errorThrown);
            });
    });
}

function fetchMasterAffiliateEarningsReport(rukoUserId, filters = null) {
    const settings = getAjaxSettingsProxy('/reports/MasterAffiliate/EarningsReport', rukoUserId, filters);

    return new Promise((resolve, reject) => {
        $.ajax(settings)
            .done((response) => resolve(response))
            .fail((jqXHR, textStatus, errorThrown) => {
                console.error('Request failed:', textStatus, errorThrown);
                console.error('Server response:', jqXHR.responseText);
                reject(errorThrown);
            });
    });
}

function fetchMasterAffiliateAdvertiserEarningsReport(rukoUserId, filters = null) {
    const settings = getAjaxSettingsProxy('/reports/MasterAffiliate/AdvertiserEarningsReport', rukoUserId, filters);

    return new Promise((resolve, reject) => {
        $.ajax(settings)
            .done((response) => resolve(response))
            .fail((jqXHR, textStatus, errorThrown) => {
                console.error('Request failed:', textStatus, errorThrown);
                console.error('Server response:', jqXHR.responseText);
                reject(errorThrown);
            });
    });
}



// @Async Load
async function loadInitialData(rukoUserId) {
    try {
        const response = await fetchIsMasterAccount(rukoUserId);

        loadTabs(rukoUserId, response.IsMasterAccount);
        loadMain();
    } catch (error) {
        loadError(error);
    } finally {
        // console.log('# Loaded - Initial Data #');
    }
}

async function loadReportFilters(rukoUserId, type, earnings = false) {
    try {
        const response = await fetchReportFilters(rukoUserId, type);
        if (!response || !response.filters) return;

        if (type === 'earnings') {
            loadEarningsInfo(response.filters.payment_frequency);
            loadFiltersEarnings(response.filters.timeframes, response.filters.payment_frequency);

            return { payment_frequency: response.filters.payment_frequency, timeframes: response.filters.timeframes }
        }
        else if (type === 'master_affiliate' && earnings) {
            loadFiltersNetworkEarnings(response.filters.timeframes, response.filters.payment_frequency);

            if (response.filters.sub_accounts && response.filters.sub_accounts.length > 0) loadFilters(response.filters.sub_accounts, 'sub_accounts');

            loadFiltersButtons();
            loadFiltersAction();

            return { payment_frequency: response.filters.payment_frequency, timeframes: response.filters.timeframes }
        }
        else {
            if (type === 'summary') loadPeriodSelectorSummary();
            else loadPeriodSelector(type);

            if (type === 'master_affiliate' && response.filters.sub_accounts && response.filters.sub_accounts.length > 0) loadFilters(response.filters.sub_accounts, 'sub_accounts');
            else if (response.filters.traffic_source && response.filters.traffic_source.length > 0) loadFilters(response.filters.traffic_source, 'traffic_source');

            if (response.filters.advertiser && response.filters.advertiser.length > 0) loadFilters(response.filters.advertiser, 'advertiser');
            if (response.filters.affiliate_link && response.filters.affiliate_link.length > 0) loadFilters(response.filters.affiliate_link, 'affiliate_link');

            loadFiltersButtons();
            loadFiltersAction();
        }
    } catch (error) {
        loadError(error);
    } finally {
        // console.log('# Loaded - Report Filters #');
    }
}

async function loadSummaryReport(rukoUserId) {
    try {
        const response = await fetchSummaryReport(rukoUserId);
        const formattedRecords = formatRecords(response.records);

        await loadReportFilters(rukoUserId, 'summary');

        loadKeyMetrics('summary', formattedRecords);

        if (formattedRecords) {
            loadChartSection();
            loadChart('line', formattedRecords);
            loadTable('summary', formattedRecords);
        }
    } catch (error) {
        loadError(error);
    } finally {
        $('#report_page_content').removeClass('loading');
        // console.log('# Loaded - Summary Report #');
    }
}

async function loadTrafficReport(rukoUserId, time) {
    try {
        const response = await fetchTrafficReport(rukoUserId, time);
        const formattedRecords = formatRecords(response.records);

        await loadReportFilters(rukoUserId, 'traffic');

        if (formattedRecords) {
            loadTable('traffic', formattedRecords, 'Detailed View', time);
        }
    } catch (error) {
        loadError(error);
    } finally {
        // console.log('# Loaded - Traffic Report #');
    }
}

async function loadAdvertiserReport(rukoUserId, time) {
    try {
        const response = await fetchAdvertiserReport(rukoUserId, time);
        const formattedRecords = formatRecords(response.records, 'advertiser');

        await loadReportFilters(rukoUserId, 'advertiser');

        if (formattedRecords) {
            loadTable('advertiser', formattedRecords, 'Detailed View', time);
        }
    } catch (error) {
        loadError(error);
    } finally {
        // console.log('# Loaded - Advertiser Report #');
    }
}

async function loadTrafficSourceReport(rukoUserId, time) {
    try {
        const response = await fetchTrafficSourceReport(rukoUserId, time);
        const formattedRecords = formatRecords(response.records, 'traffic_source');

        await loadReportFilters(rukoUserId, 'traffic_source');

        if (formattedRecords) {
            loadTable('trafficSource', formattedRecords, 'Detailed View', time);
        }
    } catch (error) {
        loadError(error);
    } finally {
        // console.log('# Loaded - Advertiser Report #');
    }
}

async function loadEarningsReport(rukoUserId) {
    try {
        const today = moment().format('YYYY-MM-DD');

        const startDate = '2000-01-01';
        const endDate = today;

        const response = await fetchEarningsReport(rukoUserId, { startDate, endDate });

        if (response.totalRecords === 0) {
            loadError('No records found.');
            return;
        }

        const filters = await loadReportFilters(rukoUserId, 'earnings');

        const formattedRecords = formatRecordsEarnings(response.records, filters);

        loadFiltersEarningsActions(formattedRecords, filters);

        loadKeyMetrics('earnings', formattedRecords);

        if (formattedRecords) {
            loadTable('earnings', formattedRecords);
        }
    } catch (error) {
        loadError(error);
    } finally {
        // console.log('# Loaded - Advertiser Report #');
    }
}

async function loadNetworkTraffic(rukoUserId) {
    try {
        const responseTR = await fetchMasterAffiliateTrafficReport(rukoUserId);
        const responseAR = await fetchMasterAffiliateAdvertiserReport(rukoUserId);

        const formattedTR = formatRecordsMasterAffiliate(responseTR.records, 'kycStatus');
        const formattedAR = formatRecordsMasterAffiliate(responseAR.records, 'advertiser');

        await loadReportFilters(rukoUserId, 'master_affiliate');

        if (formattedTR) {
            loadTable('masterAffiliateTR', formattedTR, 'Traffic Report per Sub-Affiliate');
        }

        if (formattedAR) {
            loadTable('masterAffiliateAR', formattedAR, 'Traffic Report per Sub-Affiliate and Advertiser');
        }
    } catch (error) {
        loadError(error);
    } finally {
        // console.log('# Loaded - Advertiser Report #');
    }
}

async function loadNetworkEarnings(rukoUserId) {
    try {
        const today = moment().format('YYYY-MM-DD');

        const startDate = '2000-01-01';
        const endDate = today;

        const responseER = await fetchMasterAffiliateEarningsReport(rukoUserId, { startDate, endDate });
        const responseAR = await fetchMasterAffiliateAdvertiserEarningsReport(rukoUserId, { startDate, endDate });

        const formattedER = formatRecordsMasterAffiliateNetwork(responseER.records);
        const formattedAR = formatRecordsMasterAffiliateNetwork(responseAR.records, 'advertiser');

        const filters = await loadReportFilters(rukoUserId, 'master_affiliate', true);

        loadFiltersNetworkEarningsActions(formattedER, formattedAR, filters);

        loadKeyMetrics('networkEarnings', formattedER);

        if (formattedER) {
            loadTable('masterAffiliateNetworkER', formattedER, 'Earnings per Sub-Affiliate and Period', 'Daily', filters.payment_frequency);
        }

        if (formattedER) {
            loadTable('masterAffiliateNetworkAR', formattedAR, 'Earnings per Sub-Affiliate, Period and Advertiser', 'Daily', filters.payment_frequency);
        }
    } catch (error) {
        loadError(error);
    } finally {
        // console.log('# Loaded - Advertiser Report #');
    }
}

async function loadReportData(id, rukoUserId) {
    try {
        if (id === 'tab-traffic-report') {
            await loadTrafficReport(rukoUserId, 'Daily'); // Call Traffic Report Daily
        }
        else if (id === 'tab-advertiser-report') {
            await loadAdvertiserReport(rukoUserId, 'Daily'); // Call Advertiser Report daily
        }
        else if (id === 'tab-traffic-source-report') {
            await loadTrafficSourceReport(rukoUserId, 'Daily'); // Call Traffic Source Report Daily
        }
        else if (id === 'tab-earnings-report') {
            await loadEarningsReport(rukoUserId); // Call Earnings Report
        }
        else if (id === 'tab-network-traffic') {
            await loadNetworkTraffic(rukoUserId); // Call Network Traffic
        }
        else if (id === 'tab-network-earnings') {
            await loadNetworkEarnings(rukoUserId); // Call Network Earnings
        }
    } catch (error) {
        loadError(error);
    } finally {
        $('#report_page_content').removeClass('loading');
        // console.log('# Loaded - Report Data #');
    }
}

async function loadFiltersData(id, time, filters, frequency) {
    const RUKO_USER_ID = getRukoUserID();

    try {
        if (id === 'tab-summary') await reloadSummaryData(RUKO_USER_ID, filters);
        else if (id === 'tab-traffic-report') await reloadTrafficData(RUKO_USER_ID, filters, time);
        else if (id === 'tab-advertiser-report') await reloadAdvertiserData(RUKO_USER_ID, filters, time);
        else if (id === 'tab-traffic-source-report') await reloadTrafficSourceData(RUKO_USER_ID, filters, time);
        else if (id === 'tab-network-traffic') await reloadNetworkTraffic(RUKO_USER_ID, filters);
        else if (id === 'tab-network-earnings') await reloadNetworkEarnings(RUKO_USER_ID, filters, frequency);
    } catch (error) {
        loadError(error);
    } finally {
        $('#report_page_content').removeClass('loading');
        // console.log('# Loaded Filters #');
    }
}



// @Async Reload
async function reloadSummaryData(rukoUserId, filters) {
    const response = await fetchSummaryReport(rukoUserId, filters);
    if (!response) return;

    const formattedRecords = formatRecords(response.records);

    loadKeyMetrics('summary', formattedRecords);

    if (formattedRecords) {
        loadChartSection();
        loadChart('line', formattedRecords);
        loadTable('summary', formattedRecords);
    }
}

async function reloadTrafficData(rukoUserId, filters, time) {
    const response = await fetchTrafficReport(rukoUserId, time, filters);
    if (!response) return;

    const formattedRecords = formatRecords(response.records);

    if (formattedRecords) {
        loadTable('traffic', formattedRecords, 'Detailed View', time);
    }
}

async function reloadAdvertiserData(rukoUserId, filters, time) {
    const response = await fetchAdvertiserReport(rukoUserId, time, filters);
    if (!response) return;

    const formattedRecords = formatRecords(response.records, 'advertiser');

    if (formattedRecords) {
        loadTable('advertiser', formattedRecords, 'Detailed View', time);
    }
}

async function reloadTrafficSourceData(rukoUserId, filters, time) {
    const response = await fetchTrafficSourceReport(rukoUserId, time, filters);
    if (!response) return;

    const formattedRecords = formatRecords(response.records, 'traffic_source');

    if (formattedRecords) {
        loadTable('trafficSource', formattedRecords, 'Detailed View', time);
    }
}

async function reloadNetworkTraffic(rukoUserId, filters) {
    const responseTR = await fetchMasterAffiliateTrafficReport(rukoUserId, filters);
    const responseAR = await fetchMasterAffiliateAdvertiserReport(rukoUserId, filters);

    const formattedTR = formatRecordsMasterAffiliate(responseTR.records, 'kycStatus');
    const formattedAR = formatRecordsMasterAffiliate(responseAR.records, 'advertiser');

    if (formattedTR) {
        loadTable('masterAffiliateTR', formattedTR, 'Traffic Report per Sub-Affiliate');
    }

    if (formattedAR) {
        loadTable('masterAffiliateAR', formattedAR, 'Traffic Report per Sub-Affiliate and Advertiser');
    }
}

async function reloadNetworkEarnings(rukoUserId, filters, frequency) {
    const today = moment().format('YYYY-MM-DD');

    const startDate = '2000-01-01';
    const endDate = today;

    filters.startDate = startDate;
    filters.endDate = endDate;

    const responseER = await fetchMasterAffiliateEarningsReport(rukoUserId, filters);
    const responseAR = await fetchMasterAffiliateAdvertiserEarningsReport(rukoUserId, filters);

    const formattedER = formatRecordsMasterAffiliateNetwork(responseER.records);
    const formattedAR = formatRecordsMasterAffiliateNetwork(responseAR.records, 'advertiser');

    const options = $('#tab-network-earnings .filters-menu .filter.network-earnings').find('.option');
    $(options).addClass('active');

    loadKeyMetrics('networkEarnings', formattedER);

    if (formattedER) {
        loadTable('masterAffiliateNetworkER', formattedER, 'Earnings per Sub-Affiliate and Period', 'Daily', frequency);
    }

    if (formattedAR) {
        loadTable('masterAffiliateNetworkAR', formattedAR, 'Earnings per Sub-Affiliate, Period and Advertiser', 'Daily', frequency);
    }
}



// @Main
$(function () {
    // timeout is the only way to detect this specific page 
    setTimeout(function () {
        if ($('#report_page_1').length) {
            const RUKO_USER_ID = getRukoUserID();

            // Load hardcoded Data
            if ($('#report_page_content').length) loadPage();

            // Load Initial Data
            loadInitialData(RUKO_USER_ID);

            // Call Summary Report
            loadSummaryReport(RUKO_USER_ID);
        }
    }, 500);
});






// @CUSTOM CALENDAR

function createCalendarDays(id, afterRangeSelected) {
    var adjusting = false; // guard flag to avoid recursion

    new AirDatepicker('#' + id, {
        locale: enUS,
        range: true,
        toggleSelected: false,
        onSelect: function ({ date, datepicker }) {
            if (adjusting) return;
            var dates = Array.isArray(date) ? date : (date ? [date] : []);

            // Normalize single-day range to [d, d] - If user picked only one day, force it to [d, d] WITHOUT re-firing onSelect
            if (dates.length === 1) {
                adjusting = true;
                datepicker.selectDate([dates[0], dates[0]], { silent: true });
                adjusting = false;
            }

            // Fire custom callback only when the range is complete (2 dates)
            var sel = datepicker.selectedDates;
            if (sel && sel.length === 2 && typeof afterRangeSelected === 'function') {
                var start = sel[0];
                var end = sel[1];

                // Provide both Date objects and formatted strings
                afterRangeSelected({
                    start: start,
                    end: end,
                    startISO: datepicker.formatDate(start, 'yyyy-MM-dd'),
                    endISO: datepicker.formatDate(end, 'yyyy-MM-dd')
                });
                return;
            }
        }
    });
}

function createCalendarMonths(id, afterRangeSelected) {
    let adjusting = false; // guard flag to avoid recursion

    new AirDatepicker(`#${id}`, {
        locale: enUS,
        view: 'months',
        minView: 'months',
        range: true,
        toggleSelected: false,
        // optional: autoClose: true, clearButton: true
        onSelect({ date, datepicker }) {
            if (adjusting) return;

            const dates = Array.isArray(date) ? date : (date ? [date] : []);

            // Normalize single-month range to [m, m] silently (without re-firing onSelect)
            if (dates.length === 1) {
                adjusting = true;
                datepicker.selectDate([dates[0], dates[0]], { silent: true });
                adjusting = false;
            }

            const sel = datepicker.selectedDates;
            if (!sel || sel.length !== 2) return;

            // Ensure chronological order
            const a = sel[0], b = sel[1];
            const first = new Date(Math.min(a, b));
            const last = new Date(Math.max(a, b));

            // Expand to the first day of the first month and last day of the last month
            const start = new Date(first.getFullYear(), first.getMonth(), 1);
            const end = new Date(last.getFullYear(), last.getMonth() + 1, 0);

            if (typeof afterRangeSelected === 'function') {
                afterRangeSelected({
                    start,
                    end,
                    startISO: datepicker.formatDate(start, 'yyyy-MM-dd'),
                    endISO: datepicker.formatDate(end, 'yyyy-MM-dd'),
                    startMonth: start.getMonth(),   // 0-11
                    startYear: start.getFullYear(),
                    endMonth: end.getMonth(),       // 0-11
                    endYear: end.getFullYear()
                });
            }
        }
    });
}


// @Handlers
function handleCalendarDaysAfterSelect(datepicker) {
    const activeOption = $('#report_page_content')
        .find('.report-item.active .filters-menu .active-option[data-name="period-selector"]')[0];

    if (activeOption) {
        activeOption.dataset.type = 'custom-date';
        activeOption.dataset.value = JSON.stringify({ startDate: datepicker.startISO, endDate: datepicker.endISO });

        const customSelect = $(activeOption).parent().find('.custom-select.active')[0];

        if (customSelect) {
            customSelect.dataset.type = 'custom-date';
            customSelect.dataset.value = JSON.stringify({ startDate: datepicker.startISO, endDate: datepicker.endISO });

            if (datepicker.startISO === datepicker.endISO) $(customSelect).find('span').text(datepicker.startISO);
            else $(customSelect).find('span').text(`${datepicker.startISO} - ${datepicker.endISO}`);
        }
        else {
            if (datepicker.startISO === datepicker.endISO) $(activeOption).find('span').text(datepicker.startISO);
            else $(activeOption).find('span').text(`${datepicker.startISO} - ${datepicker.endISO}`);
        }
    }

    if (window.innerWidth > 1024) $(activeOption).closest('.filter').removeClass('open');
}

function handleCalendarMonthsAfterSelect(datepicker) {
    let label;
    if (datepicker.startMonth !== datepicker.endMonth) label = `${getMonthLabelShort(datepicker.startMonth + 1)} ${datepicker.startYear} - ${getMonthLabelShort(datepicker.endMonth + 1)} ${datepicker.endYear}}`
    else label = `${getMonthLabelShort(datepicker.startMonth + 1)} - ${datepicker.startYear}`;

    const activeOption = $('#report_page_content')
        .find('.report-item.active .filters-menu .active-option[data-name="period-selector"]')[0];

    if (activeOption) {
        activeOption.dataset.type = 'custom-date';
        activeOption.dataset.value = JSON.stringify({ startDate: datepicker.startISO, endDate: datepicker.endISO });

        const customSelect = $(activeOption).parent().find('.custom-select.active')[0];

        if (customSelect) {
            customSelect.dataset.type = 'custom-date';
            customSelect.dataset.value = JSON.stringify({ startDate: datepicker.startISO, endDate: datepicker.endISO });

            $(customSelect).find('span').text(label);
        }
        else {
            $(activeOption).find('span').text(label);
        }
    }

    if (window.innerWidth > 1024) $(activeOption).closest('.filter').removeClass('open');
}


// @Utils
function getMonthLabel(month) {
    const labels = [
        'January', 'February', 'March', 'April', 'May', 'June',
        'July', 'August', 'September', 'October', 'November', 'December'
    ];
    return labels[month - 1];
}

function getMonthLabelShort(month) {
    const labels = [
        'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
        'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'
    ];
    return labels[month - 1];
}
